/*
 * @(#)AbstractRelationship.java  1.0  2. Dezember 2003
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.simulation;

import java.io.IOException;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.util.*;

/**
 * AbstractRelationship.
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 2. Dezember 2003  Created.
 */
public abstract class AbstractRelationship extends AbstractObject
implements SimulatedRelationship, SimulatedObjectListener {
    private SimulatedElement start;
    private SimulatedElement end;
    /**
     * This value is used to observe simulated link activations.
     * The value is incremented when the message activate is received,
     * and decremented when the message deactivate is received.
     */
    private int activations = 0;
    /**
     * A percentage indicating the "growth" of the relationship.
     * As seen from the source to the target.
     * Ranges from 0 to 100. 0 equals to now growth. 100 equals to a fully
     * grown relationship.
     */
    private int growth = 0;
    
    /** Creates a new instance. */
    public AbstractRelationship() {
    }
    
    public SimulatedElement getConnected(SimulatedElement startOrEnd) {
        return (startOrEnd == start) ? end : start;
    }
    
    public SimulatedElement getEnd() {
        return end;
    }
    
    public SimulatedElement getStart() {
        return start;
    }
    
    public void remap(java.util.Map oldToNew) {
        setStart((SimulatedElement) oldToNew.get(getStart()));
        setEnd((SimulatedElement) oldToNew.get(getEnd()));
    }
    
    public void setEnd(SimulatedElement newEnd) {
        if (newEnd != end) {
            if (end != null) {
                end.removeSimulatedObjectListener(this);
                if (start != null) {
                    handleDisconnecting(start, end);
                    start.removeRelationship(this);
                    end.removeRelationship(this);
                }
            }
            end = newEnd;
            if (end != null) {
                end.addSimulatedObjectListener(this);
                if (start != null) {
                    handleConnecting(start, end);
                    start.addRelationship(this);
                    end.addRelationship(this);
                }
            }
        }
    }
    
    public void setStart(SimulatedElement newStart) {
        if (newStart != start) {
            if (start != null) {
                start.removeSimulatedObjectListener(this);
                if (end != null) {
                    handleDisconnecting(start, end);
                    start.removeRelationship(this);
                    end.removeRelationship(this);
                    handleDisconnected(start, end);
                }
            }
            start = newStart;
            if (start != null) {
                start.addSimulatedObjectListener(this);
                if (end != null) {
                    handleConnecting(start, end);
                    start.addRelationship(this);
                    end.addRelationship(this);
                    handleConnected(start, end);
                }
            }
        }
    }
    
    /**
     * Handles the disconnection of a relationship.
     * Override this method to handle this event.
     */
    protected void handleDisconnecting(SimulatedElement start, SimulatedElement end) {
        
    }
    /**
     * Handles the disconnection of a relationship.
     * Override this method to handle this event.
     */
    protected void handleDisconnected(SimulatedElement start, SimulatedElement end) {
        
    }
    
    /**
     * Handles the relationship of a relationship.
     * Override this method to handle this event.
     */
    protected void handleConnecting(SimulatedElement start, SimulatedElement end) {
    }
    
    /**
     * Handles the relationship of a relationship.
     * Override this method to handle this event.
     */
    protected void handleConnected(SimulatedElement start, SimulatedElement end) {
    }
    
    
    public void objectAdded(SimulatedObjectEvent evt) {
    }
    
    public void objectRemoved(SimulatedObjectEvent evt) {
        /*
        if (evt.getElement() == getStart()) {
            setStart(null);
        }
        if (evt.getElement() == getEnd()) {
            setEnd(null);
        }*/
        fireObjectRequestRemove();
    }
    
    public void elementRenamed(SimulatedObjectEvent evt) {
    }
    
    public void objectRequestRemove(SimulatedObjectEvent evt) {
    }
    
    public void objectChanged(SimulatedObjectEvent evt) {
    }
    
    /**
     * Creates a shallow copy of this element.
     */
    public AbstractRelationship clone() {
        AbstractRelationship that = (AbstractRelationship) super.clone();
        if (that.start != null) {
            that.start.addSimulatedObjectListener(that);
            that.start.addRelationship(that);
        }
        if (that.end != null) {
            that.end.addSimulatedObjectListener(that);
            that.end.addRelationship(that);
        }
        if (that.start != null && that.end != null) {
            that.handleConnecting(start, end);
        }
        return that;
    }
    
    public String toString() {
        return super.toString()+"["+getStart()+","+getEnd()+"]";
    }
    
    public void relationshipAdded(SimulatedObjectEvent e) {
    }
    
    public void relationshipRemoved(SimulatedObjectEvent e) {
    }
    
    /**
     * Activates this relationship.
     * A relationship may be activated multiple times.
     */
    public void activate() {
        activations++;
        fireStateChanged();
    }
    /**
     * Deactivates this relationship.
     * A relationship may be deactivated as many times as it has been activated.
     */
    public void deactivate() {
        activations--;
        fireStateChanged();
    }
    /**
     * Sets the growth of the relationship.
     *
     * @param percentage A percentage ranging from 0 to 100. 0 equals to no
     * growth, 100 equals to a fully grown relationship.
     */
    public void setGrowth(int percentage) {
        if (percentage != growth) {
            growth = percentage;
            fireStateChanged();
        }
    }
    
    /**
     * Returns true if this relationship is active.
     */
    public boolean isActive() {
        return activations != 0;
    }
    
    /**
     * Returns the growth of the relationship.
     *
     * @return A percentage ranging from 0 to 100. 0 equals to no
     * growth, 100 equals to a fully grown relationship.
     */
    public int getGrowth() {
        return growth;
    }
    
    /**
     * Let's the relationship "grow" from the start to the end of the relationship.
     */
    public void grow() {
        if (getSimulation().getDelay() == 0) {
            setGrowth(100);
            
        } else {
            activate();
            for (int i=0; i <= 100; i += 2) {
                try { Thread.sleep(Math.max(1, getSimulation().getDelay() / 100)); } catch (InterruptedException e) {}
                setGrowth(i);
            }
            deactivate();
        }
    }
    /**
     * Let's the relationship "shrink" from the end to the start of the relationship.
     */
    public void shrink() {
        if (getSimulation().getDelay() == 0) {
            setGrowth(0);
            
        } else {
            activate();
            for (int i=100; i >= 0; i -= 2) {
                try { Thread.sleep(Math.max(1, getSimulation().getDelay() / 100)); } catch (InterruptedException e) {}
                setGrowth(i);
            }
            deactivate();
        }
    }
    
    public void write(DOMOutput out) throws IOException {
        out.openElement("start");
        out.writeObject(getStart());
        out.closeElement();
        
        out.openElement("end");
        out.writeObject(getEnd());
        out.closeElement();
    }
    
    public void read(DOMInput in) throws IOException {
        in.openElement("start");
        setStart((SimulatedElement) in.readObject());
        in.closeElement();
        
        in.openElement("end");
        setEnd((SimulatedElement) in.readObject());
        in.closeElement();
    }
    
    public void removeNotify(Simulation simulation) {
        // XXX - Maybe we need can't do setStart(null), setEnd(null) here.
        // But we need this (at least for the PertModel).
        setStart(null);
        setEnd(null);
        
        super.removeNotify(simulation);
    }
}
